#pragma once

#ifndef _TOKEN_H
#define _TOKEN_H
#include <iostream>
#include "list.h"
#include "color.h"

using namespace std;

/*
 * This class creates a token list from a formula string.
 *
 * It is a FSM using pre-created state classes, that need to be
 * created before the FSM is inii'ed.
 */

//class StateBase;
class Tokenize;
class StateBase;
extern StateBase *state_start;		// This is so state_start can be set at FSM init.


class StateBase {
	protected:
	public:
		virtual void enter (Tokenize *) = 0;
		virtual void work (Tokenize *, const char *) = 0;
		virtual void exit (Tokenize *) = 0;
};

typedef struct {
	struct Node node;
	int type;
	union {
		double number;
		long long integer;
		char *str;
	};
} t_node;

enum {NT_INT = 1, NT_FLOAT, NT_STRING, NT_ADD, NT_SUB, NT_MULT, NT_OPEN, NT_CLOSE, NT_DIV};
static const char *nt_name[] = {"","INT","FLOAT","STR","ADD","SUB","MULT","OPEN","CLOSE","DIV"};

extern void create_all_states (void);
extern void delete_all_states (void);

class Tokenize {
	private:
		string name = "Tokenize";		// for debug
		void reset (void) { flush_list(); formula = ""; used = 0; stop = false; } 
		void print_node (t_node *node) {		/* debug */
			cout << nt_name[node->type];
			switch (node->type) {
				case NT_INT: { cout << ":" << node->integer; break; }
				case NT_FLOAT: { cout << ":" << node->number; break; }
				case NT_STRING: { cout << ":\"" << node->str << "\"";  break; }
			}
		}
		void flush_list (void) {
			t_node *node;
			while (node = (t_node *)list->rem_head()) {
				print_node (node);
				cout << "\n";
				free (node);
			}
		}
	protected:
		StateBase *state;
	public:
		List *list;
		string formula;
		int used = 0;
		bool stop = false;
		Tokenize(List *list) { 
			create_all_states(); 
			cout << "Tokenize::Tokenize() " << state_start << "\n"; 
			this->list = list; 
			this->state = state_start; 
		}
		~Tokenize() { cout << "destroy\n"; delete_all_states(); }
		void set_formula (string str) { reset(); formula = str; }
		void print_list (void) {				/* debug */
			cout << name << " list:" << list->length() << " - ";
			for (t_node *node = (t_node *)list->get_head(); node->node.ln_next; node = (t_node *)node->node.ln_next) {
				print_node(node);
				cout << " ";
			}
			cout << "\n";
		}
		void set_state (StateBase *n) { 
			cout << name << " set_state()\n";  
			if (state) state->exit(this);
			state = n;
			if (state) state->enter(this);
		}
		bool do_work (void) { 
			cout << name << " do_work()  ";
			//int loops = 60; 
			//while (--loops > 0 && !stop && (formula.length() > 0)) {
			//	cout << BLUE << "<" << loops << "> " << RESET;
			while (!stop && (formula.length() > 0)) {
				cout << COLOR_BLUE << "<" << formula.length() << "> " << COLOR_RESET;
				state->work(this, &formula.c_str()[used]);
			} 
			return formula.length() == 0;
		};
};

class StateStart : public StateBase {
	protected:
		string name = "Start";
	public:
		void enter (Tokenize *fsm) override { cout << name << ":enter()\n"; };
		void work (Tokenize *fsm, const char *str) override;
		void exit (Tokenize *fsm) override { cout << name << ":exit()\n"; };
};

/* create an integer node in the list */
static void create_integer_node (Tokenize *fsm, long long integer) {
	t_node *node = (t_node *) malloc (sizeof(t_node));
	node->type = NT_INT;
	node->integer = integer;
	fsm->list->add_tail (&node->node);
	fsm->formula = fsm->formula.substr(fsm->used);
	fsm->used = 0;
}

/* create a node that uses no extra data, (e.g. a node for '*' ) */
static void create_typed_node (Tokenize *fsm, int type) {
	t_node *node = (t_node *) malloc (sizeof(t_node));
	node->type = type;
	fsm->list->add_tail (&node->node);
	fsm->formula = fsm->formula.substr(fsm->used);
	fsm->used = 0;
}

class StateNumber : public StateBase {
	protected:
		bool save = true;
		string name = "Number";
	public:
		void enter (Tokenize *fsm) override { cout << name << ":enter()\n"; save = true; };
		void work (Tokenize *fsm, const char *str) override;
		void exit (Tokenize *fsm) override;
};

class StateAddSub : public StateBase {
	protected:
		bool save;
		int type;
		string name = "Add Sub";
	public:
		void enter (Tokenize *fsm) override { cout << name << ":enter()\n"; save = true; };
		void work (Tokenize *fsm, const char *str) override;
		void exit (Tokenize *fsm) override;
};

class StateFloat : public StateBase {
	protected:
	string name = "Float";
	public:
		void enter (Tokenize *fsm) override { cout << name << ":enter()\n"; };
		void work (Tokenize *fsm, const char *str) override;
		void exit (Tokenize *fsm) override;
};

class StateSingleChar : public StateBase {
	protected:
	string name = "SingleChar";
	char find = 0;
	int nt_type = 0;
	public:
		StateSingleChar (const char ch, int type) { 
			find = ch; 
			name += "("; 
			name += ch; 
			name += ")"; 
			nt_type = type; 
		} 
		void enter (Tokenize *fsm) override { cout << name << ":enter()\n"; };
		void work (Tokenize *fsm, const char *str) override;
		void exit (Tokenize *fsm) override;
};

#endif

